<?php
namespace App\Models;

use App\Core\Model;
use PDO;

class User extends Model
{
    public function findByTelegramId(string $telegramId): ?array
    {
        $stmt = $this->db->prepare('SELECT * FROM users WHERE telegram_id = ? LIMIT 1');
        $stmt->execute([$telegramId]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public function create(array $data): int
    {
        $stmt = $this->db->prepare('INSERT INTO users (telegram_id, username, first_name, last_name, subscription_expiry, status, created_at) VALUES (:telegram_id, :username, :first_name, :last_name, :subscription_expiry, :status, NOW())');
        $stmt->execute([
            ':telegram_id' => $data['telegram_id'],
            ':username' => $data['username'],
            ':first_name' => $data['first_name'],
            ':last_name' => $data['last_name'],
            ':subscription_expiry' => $data['subscription_expiry'] ?? null,
            ':status' => $data['status'] ?? 'active'
        ]);
        return (int)$this->db->lastInsertId();
    }

    public function updateSubscription(int $userId, ?string $expiry): void
    {
        $stmt = $this->db->prepare('UPDATE users SET subscription_expiry = :expiry WHERE id = :id');
        $stmt->execute([':expiry' => $expiry, ':id' => $userId]);
    }

    public function updateStatus(int $userId, string $status): void
    {
        $stmt = $this->db->prepare('UPDATE users SET status = :status WHERE id = :id');
        $stmt->execute([':status' => $status, ':id' => $userId]);
    }

    public function allWithPagination(int $limit = 50, int $offset = 0): array
    {
        $stmt = $this->db->prepare('SELECT * FROM users ORDER BY id DESC LIMIT :limit OFFSET :offset');
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    public function count(): int
    {
        return (int)$this->db->query('SELECT COUNT(*) FROM users')->fetchColumn();
    }

    public function countActiveSubscriptions(): int
    {
        $stmt = $this->db->query("SELECT COUNT(*) FROM users WHERE subscription_expiry >= NOW() AND status = 'active'");
        return (int)$stmt->fetchColumn();
    }

    public function countExpiredSubscriptions(): int
    {
        $stmt = $this->db->query("SELECT COUNT(*) FROM users WHERE (subscription_expiry IS NULL OR subscription_expiry < NOW()) AND status = 'active'");
        return (int)$stmt->fetchColumn();
    }

    public function countBlocked(): int
    {
        $stmt = $this->db->query("SELECT COUNT(*) FROM users WHERE status = 'blocked'");
        return (int)$stmt->fetchColumn();
    }

    public function findById(int $id): ?array
    {
        $stmt = $this->db->prepare('SELECT * FROM users WHERE id = ? LIMIT 1');
        $stmt->execute([$id]);
        $row = $stmt->fetch();
        return $row ?: null;
    }
}
