<?php
namespace App\Models;

use App\Core\Model;
use PDO;

class Ticket extends Model
{
    public function openTicket(int $userId): int
    {
        $code = strtoupper(bin2hex(random_bytes(4)));
        $stmt = $this->db->prepare('INSERT INTO tickets (ticket_code, user_id, status, created_at) VALUES (?, ?, "open", NOW())');
        $stmt->execute([$code, $userId]);
        return (int)$this->db->lastInsertId();
    }

    public function findOpenByUser(int $userId): ?array
    {
        $stmt = $this->db->prepare('SELECT * FROM tickets WHERE user_id = ? AND status = "open" ORDER BY id DESC LIMIT 1');
        $stmt->execute([$userId]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public function findByCode(string $code): ?array
    {
        $stmt = $this->db->prepare('SELECT * FROM tickets WHERE ticket_code = ?');
        $stmt->execute([$code]);
        return $stmt->fetch() ?: null;
    }

    public function findById(int $id): ?array
    {
        $stmt = $this->db->prepare('SELECT * FROM tickets WHERE id = ?');
        $stmt->execute([$id]);
        return $stmt->fetch() ?: null;
    }

    public function assign(int $ticketId, int $staffId): void
    {
        $stmt = $this->db->prepare('UPDATE tickets SET assigned_staff_id = ?, status = "open" WHERE id = ?');
        $stmt->execute([$staffId, $ticketId]);
    }

    public function close(int $ticketId): void
    {
        $stmt = $this->db->prepare('UPDATE tickets SET status = "closed", closed_at = NOW() WHERE id = ?');
        $stmt->execute([$ticketId]);
    }

    public function listAll(int $limit = 50): array
    {
        $stmt = $this->db->prepare('SELECT t.*, s.name AS staff_name FROM tickets t LEFT JOIN staff s ON t.assigned_staff_id = s.id ORDER BY t.id DESC LIMIT :limit');
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    public function listForStaff(int $staffId, int $limit = 50): array
    {
        $stmt = $this->db->prepare('SELECT * FROM tickets WHERE assigned_staff_id = :staff_id ORDER BY id DESC LIMIT :limit');
        $stmt->bindValue(':staff_id', $staffId, PDO::PARAM_INT);
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    public function stats(): array
    {
        return [
            'total' => (int)$this->db->query('SELECT COUNT(*) FROM tickets')->fetchColumn(),
            'open' => (int)$this->db->query('SELECT COUNT(*) FROM tickets WHERE status="open"')->fetchColumn(),
            'closed' => (int)$this->db->query('SELECT COUNT(*) FROM tickets WHERE status="closed"')->fetchColumn(),
        ];
    }

    public function markFirstReplyIfNeeded(int $ticketId): void
    {
        $stmt = $this->db->prepare('UPDATE tickets SET first_reply_at = NOW() WHERE id = ? AND first_reply_at IS NULL');
        $stmt->execute([$ticketId]);
    }

    public function avgReplySeconds(): float
    {
        $stmt = $this->db->query('SELECT AVG(TIMESTAMPDIFF(SECOND, created_at, first_reply_at)) AS avg_seconds FROM tickets WHERE first_reply_at IS NOT NULL');
        return (float)$stmt->fetchColumn();
    }

    public function avgResolutionSeconds(): float
    {
        $stmt = $this->db->query('SELECT AVG(TIMESTAMPDIFF(SECOND, created_at, closed_at)) AS avg_seconds FROM tickets WHERE closed_at IS NOT NULL');
        return (float)$stmt->fetchColumn();
    }
}
